#ifndef WaveBump
	#define WaveBump 0.8
#endif
#ifndef WaveScale
	#define WaveScale 1
#endif

#define BaseOpacity 0.95
//#define InvertWaves

struct VS_INPUT
{
    float4 Position   : POSITION;
    float2 TexCoord   : TEXCOORD0;
};

struct VS_OUTPUT_MAIN
{
    float4 Position 	: POSITION;
    float2 TexCoord0	: TEXCOORD0;
    float4 TexCoord1	: TEXCOORD1;
	float3 ViewDepth	: TEXCOORD2;
	float4 WorldPos		: TEXCOORD3;
};

struct VS_OUTPUT_SHORE
{
    float4 Position 	: POSITION;
    float2 TexCoord 	: TEXCOORD0;
	float FogAmount		: TEXCOORD1;
};

sampler2D Tex0 : register( s0 );
sampler2D Tex1 : register( s1 );

samplerCUBE TexEnv : register( s4 );
sampler2D TexNormal : register( s5 );
sampler2D TexNormalPrev : register( s6 );

sampler2D TexRefl : register( s7 );

float4x4 World;
float4x4 View;
float4x4 ViewProj;

float4x4 TexTx0;
float4x4 TexTx1;

float fNormalBlend;

float4 TFactor;

// FOG 
bool bFogEnabled;
float fFogEnd;
float fFogStart;
float3 vFogColor;

float CalcFog(float depth)
{
	return bFogEnabled ? saturate((depth - fFogStart)/(fFogEnd - fFogStart)) : 0.0;
}

void ApplyFog(inout float4 color, float fogAmount)
{
	color.rgb = lerp(color.rgb, vFogColor.rgb, fogAmount);
}

float4 vSunDir;
float3 vAmbient;
float3 vDiffuse;

VS_OUTPUT_MAIN vs_main(in VS_INPUT In)
{
    VS_OUTPUT_MAIN Out;
	float4 posWorld = mul(In.Position, World);
    Out.Position = mul(posWorld, ViewProj); // position in clip space
	Out.WorldPos = posWorld; // position in world space
	Out.ViewDepth = mul(posWorld, View).z; // depth of position in view space
	
	float4 TC = float4(In.TexCoord, 1.0, 1.0);
    Out.TexCoord0 = mul(TC, TexTx0).xy;
    Out.TexCoord1 = mul(TC, TexTx1).xyxy;
	
	// Make waves more prolonged
	Out.TexCoord1.x *= 0.5;
	// Store adjusted coordinates
	Out.TexCoord1 *= float4(WaveScale, WaveScale, 2.3, 1.7);

	return Out;
}

float4 ps_main(in VS_OUTPUT_MAIN In) : COLOR
{
	float4 Color;
	
	// Main waves
	float3 N1c = tex2D(TexNormal, In.TexCoord1.xy).xyz;
	float3 N1p = tex2D(TexNormalPrev, In.TexCoord1.xy).xyz;
	float3 N1 = normalize(lerp(N1p, N1c, fNormalBlend) - 0.5).xzy;
	N1.y *= WaveBump;
	// Small ripples
	float3 N2 = normalize(tex2D(TexNormal, In.TexCoord1.zw).xyz - 0.5).xzy;
	N2.y *= WaveBump;

#ifdef InvertWaves
	N1.xz = -N1.xz;
	N2.xz = -N2.xz;
#endif

	// Combine normals (world space)
	float3 N = normalize(N1*2+N2);
	
	// View vector in world space
	float3 V = -normalize(In.WorldPos.xyz);
	
	// Fresnel term
	float fr0 = 0.01; // base reflectivity
	float fr = fr0 + pow(1-saturate(dot(N, V)), 4) * (1 - fr0);
	
	// Water background texture with slight transparency
	Color.rgb = tex2D(Tex0, In.TexCoord0).rgb ;//* TFactor.rgb;
	Color.a = BaseOpacity + fr;
	
	// Diffuse + ambient 
	float NdotL = max(0, dot(N, vSunDir.xyz) + 0.1) / 1.1;
	Color.rgb *= saturate(NdotL * vDiffuse.rgb + vAmbient.rgb);
	
	// Environment
	float3 cubeVec = reflect(-V, N);
	cubeVec.y = max(0, cubeVec.y); // never reflect below horizon
	Color.rgb = lerp(Color.rgb, texCUBE(TexEnv, cubeVec).rgb * vAmbient.rgb, fr);
	
	// Specular (using direct reflectiong instead of half vector)
	float3 R = reflect(-vSunDir.xyz, N);
	float RdotV = saturate(dot(V, R));
	float3 vSpecular = pow(vAmbient + vDiffuse, 2);
	// Strong and weak
	float spec = pow(RdotV, 400) * 2 + pow(RdotV, 15) * 0.5;
	Color.rgb += spec * vSpecular.rgb * (fr + 0.15);
	
	// Pixel fog (sea water consists very large polygons)
	ApplyFog(Color, CalcFog(In.ViewDepth));
	
	return Color;
}

VS_OUTPUT_SHORE vs_shore(in VS_INPUT In)
{
    VS_OUTPUT_SHORE Out;
	float4 posWorld = mul(In.Position, World);
    Out.Position = mul(posWorld, ViewProj); 
	Out.TexCoord = In.TexCoord;
	Out.FogAmount = CalcFog(mul(posWorld, View).z);
	return Out;
}

float4 ps_shore(in VS_OUTPUT_SHORE In) : COLOR
{
	float4 Color;
	Color = tex2D(Tex0, In.TexCoord) * TFactor;
	ApplyFog(Color, In.FogAmount);
	return Color;
}

technique Water
{
	pass Pass1
	{
		VertexShader = compile vs_3_0 vs_main();
		PixelShader = compile ps_3_0 ps_main();
	}
}

technique Shore
{
	pass Pass1
	{
		VertexShader = compile vs_3_0 vs_shore();
		PixelShader = compile ps_3_0 ps_shore();
	}
}